package kz.arta.nca_iiscon.service;

import lombok.extern.slf4j.Slf4j;

import javax.xml.namespace.QName;
import javax.xml.soap.*;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;
import java.io.ByteArrayOutputStream;
import java.util.Iterator;
import java.util.Set;

@Slf4j
public class ForwardSOAPHandler implements SOAPHandler<SOAPMessageContext> {
    private static final String IISCON_NAMESPACE = "http://schemas.letograf.kz/iiscon/bus/v1";

    @Override
    public boolean handleMessage(SOAPMessageContext context) {
        Boolean outboundProperty = (Boolean) context.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY);

        if (outboundProperty) {
            try {
                SOAPMessage soapMessage = context.getMessage();
                SOAPEnvelope envelope = soapMessage.getSOAPPart().getEnvelope();

                // Устанавливаем правильный порядок namespace в envelope
                envelope.removeNamespaceDeclaration("SOAP-ENV");
                envelope.removeNamespaceDeclaration("wsu");
                envelope.removeNamespaceDeclaration("xsd");

                // Добавляем в правильном порядке
                envelope.addNamespaceDeclaration("xsd", "http://www.w3.org/2001/XMLSchema");
                envelope.addNamespaceDeclaration("SOAP-ENV", "http://schemas.xmlsoap.org/soap/envelope/");
                envelope.addNamespaceDeclaration("wsu", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");

                // Убеждаемся, что есть Header
                SOAPHeader header = envelope.getHeader();
                if (header == null) {
                    envelope.addHeader();
                }

                SOAPBody body = envelope.getBody();

                // Находим элемент SendMessage
                SOAPElement sendMessage = (SOAPElement) body.getChildElements(
                        new QName("http://bip.bee.kz/SyncChannel/v10/Types", "SendMessage")
                ).next();

                if (sendMessage != null) {
                    // Удаляем лишние namespace из SendMessage и всех потенциальных префиксов
                    removeAllAutoGeneratedNamespaces(sendMessage);

                    sendMessage.setPrefix("");

                    // Находим request
                    SOAPElement request = (SOAPElement) sendMessage.getChildElements(
                            new QName("", "request")
                    ).next();

                    if (request != null) {
//                        request.setAttribute("xmlns", "");
                        request.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns", "");

                        // Обрабатываем data элемент
                        processDataElement(request);
                    }
                }

                soapMessage.saveChanges();

                // Логируем финальное сообщение
                if (log.isInfoEnabled()) {
                    ByteArrayOutputStream out = new ByteArrayOutputStream();
                    soapMessage.writeTo(out);
                    log.info("Outbound SOAP Message:\n{}", out.toString("UTF-8"));
                }

            } catch (Exception e) {
                log.error("Error in SOAP Handler", e);
            }
        }

        return true;
    }

    private void processDataElement(SOAPElement parent) throws SOAPException {
        processDataElement(parent, true);
    }

    private void copyChildren(SOAPElement source, SOAPElement target) throws SOAPException {
        Iterator<?> it = source.getChildElements();
        while (it.hasNext()) {
            Object obj = it.next();
            if (obj instanceof SOAPElement) {
                SOAPElement srcChild = (SOAPElement) obj;
                // создаём новый элемент с тем же именем без namespace
                SOAPElement newChild = target.addChildElement(srcChild.getLocalName());
                // копируем атрибуты
                Iterator<?> attrIt = srcChild.getAllAttributes();
                while (attrIt.hasNext()) {
                    Name name = (Name) attrIt.next();
                    newChild.addAttribute(name, srcChild.getAttributeValue(name));
                }
                // рекурсивно копируем детей
                copyChildren(srcChild, newChild);
            } else {
                target.addTextNode(obj.toString());
            }
        }
    }


    private void cleanEmptyNamespaceDeclarations(SOAPElement element) throws SOAPException {
        // Сначала обрабатываем детей рекурсивно
        Iterator<?> it = element.getChildElements();
        while (it.hasNext()) {
            Object obj = it.next();
            if (obj instanceof SOAPElement) {
                cleanEmptyNamespaceDeclarations((SOAPElement) obj);
            }
        }

        // Теперь проверяем текущий элемент
        String localName = element.getLocalName();
        if (localName != null &&
                (localName.equals("applicationId")
                        || localName.equals("applicationRegistrationData")
                        || localName.equals("callOrigin")
                        || localName.equals("applicationData"))
                && (element.getNamespaceURI() == null || element.getNamespaceURI().isEmpty())) {

            replaceWithNoNamespace(element);
        }
    }

    private void replaceWithNoNamespace(SOAPElement element) throws SOAPException {
        SOAPElement parent = element.getParentElement();
        if (parent == null) return;

        // создаем новый элемент без namespace
        SOAPElement newElement = parent.addChildElement(element.getLocalName());

        // копируем атрибуты
        Iterator<?> attrIt = element.getAllAttributes();
        while (attrIt.hasNext()) {
            Name name = (Name) attrIt.next();
            newElement.addAttribute(name, element.getAttributeValue(name));
        }

        // копируем дочерние элементы
        Iterator<?> childIt = element.getChildElements();
        while (childIt.hasNext()) {
            Object child = childIt.next();
            if (child instanceof SOAPElement) {
                newElement.addChildElement((SOAPElement) child);
            } else {
                newElement.addTextNode(child.toString());
            }
        }

        // вставляем новый элемент **перед удалением старого**, чтобы сохранить порядок
        parent.insertBefore(newElement, element);
        element.detachNode();
    }


    private void processDataElement(SOAPElement parent, boolean isTopLevel) throws SOAPException {
        Iterator iterator = parent.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;
                String localName = element.getLocalName();
                SOAPElement parentElement = element.getParentElement();
                String parentName = parentElement != null ? parentElement.getLocalName() : "";

                // Если это data элемент внутри requestData (главный data с xsi:type)
                if ("data".equals(localName) && "requestData".equals(parentName)) {
                    log.info("Found data element inside requestData - adding namespaces");
                    log.info("Data element value: {}", element.getValue());
                    log.info("Data element text content: {}", element.getTextContent());

                    // data должен быть БЕЗ префикса и без namespace URI
                    element.setPrefix("");
                    QName newName = new QName("", element.getLocalName(), "");
                    element.setElementQName(newName);

                    // Добавляем namespace declarations
                    element.addNamespaceDeclaration("xsi", "http://www.w3.org/2001/XMLSchema-instance");
                    element.addNamespaceDeclaration("s01", IISCON_NAMESPACE);

                    // Добавляем xsi:type атрибут
                    element.setAttributeNS(
                            "http://www.w3.org/2001/XMLSchema-instance",
                            "xsi:type",
                            "s01:ForwardApplication"
                    );

                    log.info("Added xsi and s01 namespaces to data element");

                    // Обрабатываем дочерние элементы data
                    processForwardApplicationChildren(element);
                }
                // Если это data элемент внутри applicationData (вложенный data БЕЗ xsi:type)
                else if ("data".equals(localName) && "applicationData".equals(parentName)) {
                    // Убираем префикс у data
                    element.setPrefix("");
                    QName newName = new QName("", element.getLocalName(), "");
                    element.setElementQName(newName);

                    // НЕ добавляем xsi:type для вложенного data
                    // Просто обрабатываем его дочерние элементы (digiSign, confirmOrder)
                    processInnerDataChildren(element);
                }

                // Рекурсивно обрабатываем детей
                processDataElement(element, false);
            }
        }
    }

    private void processInnerDataChildren(SOAPElement dataElement) throws SOAPException {
        Iterator iterator = dataElement.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;
                String localName = element.getLocalName();

                // Убираем префиксы у digiSign и confirmOrder
                element.setPrefix("");
                QName newName = new QName("", element.getLocalName(), "");
                element.setElementQName(newName);

                // Для confirmOrder устанавливаем namespace URI (default namespace)
                if ("confirmOrder".equals(localName)) {
                    QName confirmOrderQName = new QName("http://pki.gov.kz/api/ws/iiscon/wsdl", "confirmOrder", "");
                    element.setElementQName(confirmOrderQName);
                    log.info("Set namespace for confirmOrder");
                    // Убираем префиксы у всех детей confirmOrder (digiSign и др.)
                    removePrefixFromChildren(element);
                }
            }
        }
    }

    private void processForwardApplicationChildren(SOAPElement dataElement) throws SOAPException {
        Iterator iterator = dataElement.getChildElements();
        log.info("Starting to process ForwardApplication children");
        int childCount = 0;
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            childCount++;
            log.info("Child {}: class={}", childCount, obj.getClass().getName());
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;
                String localName = element.getLocalName();
                log.info("Processing child element: {}", localName);

                // Определяем, какие элементы должны иметь префикс s01:
                if (shouldHaveS01Prefix(localName)) {
                    // Устанавливаем namespace URI и префикс s01:
                    QName newName = new QName(IISCON_NAMESPACE, element.getLocalName(), "s01");
                    element.setElementQName(newName);
                    log.info("Set s01 prefix for: {}", localName);

                    // Обрабатываем statusHistory отдельно
                    if ("statusHistory".equals(localName)) {
                        processStatusHistory(element);
                    }
                } else {
                    // Для registerApplication, updateApplication - устанавливаем namespace URI но БЕЗ префикса
                    QName newName = new QName(IISCON_NAMESPACE, element.getLocalName(), "");
                    element.setElementQName(newName);
                    log.info("Set default namespace for: {}", localName);

                    // Их дочерние элементы НЕ должны иметь префикс
                    removePrefixFromChildren(element);
                }
            }
        }
    }

    private void processStatusHistory(SOAPElement statusHistoryElement) throws SOAPException {
        Iterator iterator = statusHistoryElement.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;
                String localName = element.getLocalName();

                // Если это changeApplicationStatus
                if ("changeApplicationStatus".equals(localName)) {
                    // Устанавливаем namespace URI но БЕЗ префикса (default namespace)
                    QName newName = new QName(IISCON_NAMESPACE, element.getLocalName(), "");
                    element.setElementQName(newName);
                    log.info("Set default namespace for changeApplicationStatus");

                    // Убираем префиксы у всех детей
                    removePrefixFromChildren(element);
                }
            }
        }
    }

    private void removePrefixFromChildren(SOAPElement parent) throws SOAPException {
        Iterator iterator = parent.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;

                // Удаляем namespace declarations только из ЭТОГО элемента (не рекурсивно)
                removeNamespaceDeclarationsFromElement(element);

                // Убираем префикс и namespace - используем пустой namespace URI
                element.setPrefix("");
                QName newName = new QName("", element.getLocalName(), "");
                element.setElementQName(newName);

                // Рекурсивно обрабатываем детей
                removePrefixFromChildren(element);
            }
        }
    }

    /**
     * Удаляет namespace declarations только из одного элемента (не рекурсивно)
     */
    private void removeNamespaceDeclarationsFromElement(SOAPElement element) throws SOAPException {
        String[] prefixesToRemove = {"ns2", "ns3", "ns4", "ns5", "ns6", "ns7", "ns8", "ns9", "s01", "xsi"};
        for (String prefix : prefixesToRemove) {
            try {
                element.removeNamespaceDeclaration(prefix);
            } catch (Exception e) {
                // Игнорируем, если префикс не существует
            }
        }
    }

    /**
     * Удаляет все auto-generated namespace declarations (ns2, ns3, ns4, ns5, etc.)
     */
    private void removeAllAutoGeneratedNamespaces(SOAPElement element) throws SOAPException {
        // Удаляем известные префиксы
        String[] prefixesToRemove = {"ns2", "ns3", "ns4", "ns5", "ns6", "ns7", "ns8", "ns9", "s01", "xsi"};
        for (String prefix : prefixesToRemove) {
            try {
                element.removeNamespaceDeclaration(prefix);
            } catch (Exception e) {
                // Игнорируем, если префикс не существует
            }
        }

        // Также рекурсивно удаляем из всех дочерних элементов
        Iterator iterator = element.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                removeAllAutoGeneratedNamespaces((SOAPElement) obj);
            }
        }
    }

    /**
     * Удаляет лишние xmlns="" и префиксы после обработки
     */
    private void removeExtraPrefixes(SOAPElement element) throws SOAPException {
        Iterator iterator = element.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement child = (SOAPElement) obj;
                String localName = child.getLocalName();

                // Рекурсивно обрабатываем детей
                removeExtraPrefixes(child);
            }
        }
    }

    /**
     * Определяет, какие элементы должны иметь префикс s01:
     */
    private boolean shouldHaveS01Prefix(String localName) {
        return "applicationId".equals(localName) ||
                "statusHistory".equals(localName);
    }

    @Override
    public boolean handleFault(SOAPMessageContext context) {
        return true;
    }

    @Override
    public void close(MessageContext context) {
    }

    @Override
    public Set<QName> getHeaders() {
        return null;
    }
}
