package kz.arta.nca_iiscon.util;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;

@Slf4j
public class XmlToJsonUtil {

    private static final ObjectMapper objectMapper = new ObjectMapper();

    /**
     * Извлекает XML из поля data и конвертирует в JSON
     */
    public static String extractDataAsJson(Object response) {
        try {
            // Конвертируем response в JSON
            String responseJson = objectMapper.writeValueAsString(response);
            JsonNode rootNode = objectMapper.readTree(responseJson);

            // Получаем XML строку из responseData.data
            JsonNode dataNode = rootNode.at("/responseData/data");

            if (dataNode.isMissingNode() || dataNode.isNull()) {
                log.warn("Data node not found in response");
                return "{}";
            }

            String xmlString = dataNode.asText();

            // Парсим XML и конвертируем в JSON
            return xmlToJson(xmlString);

        } catch (Exception e) {
            log.error("Error extracting data as JSON", e);
            return "{\"error\": \"Failed to parse response\"}";
        }
    }

    /**
     * Конвертирует XML строку в JSON
     */
    public static String xmlToJson(String xmlString) {
        try {
            // Парсим XML
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            DocumentBuilder builder = factory.newDocumentBuilder();

            ByteArrayInputStream input = new ByteArrayInputStream(
                    xmlString.getBytes(StandardCharsets.UTF_8)
            );
            Document doc = builder.parse(input);

            // Конвертируем в JSON
            ObjectNode jsonNode = objectMapper.createObjectNode();
            Element root = doc.getDocumentElement();

            parseElement(root, jsonNode);

            return objectMapper.writerWithDefaultPrettyPrinter()
                    .writeValueAsString(jsonNode);

        } catch (Exception e) {
            log.error("Error converting XML to JSON", e);
            return "{\"error\": \"Failed to parse XML\"}";
        }
    }

    /**
     * Рекурсивно парсит XML элемент в JSON
     */
    private static void parseElement(Element element, ObjectNode parentNode) {
        String nodeName = getCleanName(element);

        NodeList children = element.getChildNodes();
        boolean hasElementChildren = false;
        StringBuilder textContent = new StringBuilder();

        // Проверяем, есть ли дочерние элементы
        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);
            if (child.getNodeType() == Node.ELEMENT_NODE) {
                hasElementChildren = true;
            } else if (child.getNodeType() == Node.TEXT_NODE) {
                String text = child.getTextContent().trim();
                if (!text.isEmpty()) {
                    textContent.append(text);
                }
            }
        }

        if (hasElementChildren) {
            // Элемент имеет дочерние элементы
            ObjectNode childNode = objectMapper.createObjectNode();

            for (int i = 0; i < children.getLength(); i++) {
                Node child = children.item(i);
                if (child.getNodeType() == Node.ELEMENT_NODE) {
                    parseElement((Element) child, childNode);
                }
            }

            parentNode.set(nodeName, childNode);
        } else {
            // Листовой элемент - добавляем текст
            String text = textContent.toString();
            if (!text.isEmpty()) {
                parentNode.put(nodeName, text);
            } else {
                parentNode.put(nodeName, "");
            }
        }
    }

    /**
     * Убирает namespace префикс из имени элемента
     */
    private static String getCleanName(Element element) {
        String localName = element.getLocalName();
        return localName != null ? localName : element.getNodeName();
    }

    /**
     * Конвертирует XML в JSON без pretty print
     */
    public static String xmlToJsonCompact(String xmlString) {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            DocumentBuilder builder = factory.newDocumentBuilder();

            ByteArrayInputStream input = new ByteArrayInputStream(
                    xmlString.getBytes(StandardCharsets.UTF_8)
            );
            Document doc = builder.parse(input);

            ObjectNode jsonNode = objectMapper.createObjectNode();
            Element root = doc.getDocumentElement();

            parseElement(root, jsonNode);

            return objectMapper.writeValueAsString(jsonNode);

        } catch (Exception e) {
            log.error("Error converting XML to JSON", e);
            return "{\"error\": \"Failed to parse XML\"}";
        }
    }
}