package kz.arta.nca_iiscon.service;

import lombok.extern.slf4j.Slf4j;

import javax.xml.namespace.QName;
import javax.xml.soap.*;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;
import java.io.ByteArrayOutputStream;
import java.util.Iterator;
import java.util.Set;

@Slf4j
public class ForwardSOAPHandler implements SOAPHandler<SOAPMessageContext> {
    private static final String IISCON_NAMESPACE = "http://schemas.letograf.kz/iiscon/bus/v1";

    @Override
    public boolean handleMessage(SOAPMessageContext context) {
        Boolean outboundProperty = (Boolean) context.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY);

        if (outboundProperty) {
            try {
                SOAPMessage soapMessage = context.getMessage();
                SOAPEnvelope envelope = soapMessage.getSOAPPart().getEnvelope();

                // Устанавливаем правильный порядок namespace в envelope
                envelope.removeNamespaceDeclaration("SOAP-ENV");
                envelope.removeNamespaceDeclaration("wsu");
                envelope.removeNamespaceDeclaration("xsd");

                // Добавляем в правильном порядке
                envelope.addNamespaceDeclaration("xsd", "http://www.w3.org/2001/XMLSchema");
                envelope.addNamespaceDeclaration("SOAP-ENV", "http://schemas.xmlsoap.org/soap/envelope/");
                envelope.addNamespaceDeclaration("wsu", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");

                // Убеждаемся, что есть Header
                SOAPHeader header = envelope.getHeader();
                if (header == null) {
                    envelope.addHeader();
                }

                SOAPBody body = envelope.getBody();

                // Находим элемент SendMessage
                SOAPElement sendMessage = (SOAPElement) body.getChildElements(
                        new QName("http://bip.bee.kz/SyncChannel/v10/Types", "SendMessage")
                ).next();

                if (sendMessage != null) {
                    // Удаляем лишние namespace из SendMessage
                    sendMessage.removeNamespaceDeclaration("ns4");
                    sendMessage.removeNamespaceDeclaration("ns5");
                    sendMessage.removeNamespaceDeclaration("s01");
                    sendMessage.removeNamespaceDeclaration("xsi");

                    // SendMessage должен остаться без префикса
                    sendMessage.setPrefix("");

                    // Находим request
                    SOAPElement request = (SOAPElement) sendMessage.getChildElements(
                            new QName("", "request")
                    ).next();

                    if (request != null) {
                        request.setAttribute("xmlns", "");

                        // Обрабатываем data элемент
                        processDataElement(request);
                    }
                }

                soapMessage.saveChanges();

                // Логируем финальное сообщение
                if (log.isInfoEnabled()) {
                    ByteArrayOutputStream out = new ByteArrayOutputStream();
                    soapMessage.writeTo(out);
                    log.info("Outbound SOAP Message:\n{}", out.toString("UTF-8"));
                }

            } catch (Exception e) {
                log.error("Error in SOAP Handler", e);
            }
        }

        return true;
    }

    private void processDataElement(SOAPElement parent) throws SOAPException {
        processDataElement(parent, true);
    }

    private void processDataElement(SOAPElement parent, boolean isTopLevel) throws SOAPException {
        Iterator iterator = parent.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;
                String localName = element.getLocalName();
                SOAPElement parentElement = element.getParentElement();
                String parentName = parentElement != null ? parentElement.getLocalName() : "";

                // Если это data элемент внутри requestData (главный data с xsi:type)
                if ("data".equals(localName) && "requestData".equals(parentName)) {
                    // Убираем префикс у data
                    element.setPrefix("");

                    // Добавляем xmlns:xsi и xmlns:s01
                    element.addNamespaceDeclaration("xsi", "http://www.w3.org/2001/XMLSchema-instance");
                    element.addNamespaceDeclaration("s01", IISCON_NAMESPACE);

                    // Устанавливаем xsi:type
                    element.setAttributeNS(
                            "http://www.w3.org/2001/XMLSchema-instance",
                            "xsi:type",
                            "s01:ForwardApplication"
                    );

                    // Обрабатываем дочерние элементы data
                    processForwardApplicationChildren(element);
                }
                // Если это data элемент внутри applicationData (вложенный data БЕЗ xsi:type)
                else if ("data".equals(localName) && "applicationData".equals(parentName)) {
                    // Убираем префикс у data
                    element.setPrefix("");

                    // НЕ добавляем xsi:type для вложенного data
                    // Просто обрабатываем его дочерние элементы (digiSign, confirmOrder)
                    processInnerDataChildren(element);
                }

                // Рекурсивно обрабатываем детей
                processDataElement(element, false);
            }
        }
    }

    private void processInnerDataChildren(SOAPElement dataElement) throws SOAPException {
        Iterator iterator = dataElement.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;
                String localName = element.getLocalName();

                // Убираем префиксы у digiSign и confirmOrder
                element.setPrefix("");

                // Для confirmOrder добавляем namespace
                if ("confirmOrder".equals(localName)) {
                    element.setAttribute("xmlns", "http://pki.gov.kz/api/ws/iiscon/wsdl");
                }
            }
        }
    }

    private void processForwardApplicationChildren(SOAPElement dataElement) throws SOAPException {
        Iterator iterator = dataElement.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;
                String localName = element.getLocalName();
                String namespaceURI = element.getNamespaceURI();

                // Определяем, какие элементы должны иметь префикс s01:
                if (shouldHaveS01Prefix(localName)) {
                    // Устанавливаем префикс s01: для этих элементов
                    if (IISCON_NAMESPACE.equals(namespaceURI)) {
                        element.setPrefix("s01");
                    }

                    // Обрабатываем statusHistory отдельно
                    if ("statusHistory".equals(localName)) {
                        processStatusHistory(element);
                    }
                } else {
                    // Для registerApplication, updateApplication убираем префикс
                    element.setPrefix("");

                    // Их дочерние элементы НЕ должны иметь префикс
                    removePrefixFromChildren(element);
                }
            }
        }
    }

    private void processStatusHistory(SOAPElement statusHistoryElement) throws SOAPException {
        Iterator iterator = statusHistoryElement.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;
                String localName = element.getLocalName();

                // Если это changeApplicationStatus
                if ("changeApplicationStatus".equals(localName)) {
                    // Убираем все префиксы у changeApplicationStatus и его детей
                    element.setPrefix("");
                    removePrefixFromChildren(element);
                }
            }
        }
    }

    private void removePrefixFromChildren(SOAPElement parent) throws SOAPException {
        Iterator iterator = parent.getChildElements();
        while (iterator.hasNext()) {
            Object obj = iterator.next();
            if (obj instanceof SOAPElement) {
                SOAPElement element = (SOAPElement) obj;

                // Убираем префикс
                element.setPrefix("");

                // Рекурсивно обрабатываем детей
                removePrefixFromChildren(element);
            }
        }
    }

    /**
     * Определяет, какие элементы должны иметь префикс s01:
     */
    private boolean shouldHaveS01Prefix(String localName) {
        return "applicationId".equals(localName) ||
                "statusHistory".equals(localName);
    }

    @Override
    public boolean handleFault(SOAPMessageContext context) {
        return true;
    }

    @Override
    public void close(MessageContext context) {
    }

    @Override
    public Set<QName> getHeaders() {
        return null;
    }
}
