

**Endpoint:** `POST http://localhost:13010/service/nca/iiscon`

## 📌 Описание

Сервис принимает JSON-данные и при необходимости файл с фотографией.
Фото можно передать **двумя способами**:

1. Внутри `dataJson` в поле `savedPhoto` (base64-строка)
2. Как отдельный файл (`MultipartFile file`)

`dataJson` — **обязателен**.
`savedPhoto` — **необязателен**, если используется файл.

---

## Запрос

### Multipart/Form-Data параметры:

```java
@RequestPart(value = "data") String dataJson
@RequestPart(value = "file", required = false) MultipartFile file
```

### Пример `dataJson`:

```json
{
  "request": {
    "referenceNumber": "17844565151117"
  },
  "applicantNumber": "7 (777) 777-77-77",
  "savedPhoto": "/9j/4AA.." 
}
```

* `request.referenceNumber` — обязательное поле
* `applicantNumber` — номер заявителя
* `savedPhoto` — **опционально** (фото в формате Base64).
  Если `savedPhoto` передан, файл можно не отправлять.

---

## 📤 Ответ

Сервис **nca_iiscon** по факту возвращает только `data`,
но для удобства были добавлены дополнительные поля:

* `applicationId`
* `confirmOrderSign`
* `denyOrderSign`
* `digiSign`

### Примечание

При необходимости можно менять коды внутри:

* `confirmOrderSign`
* `denyOrderSign`
* `digiSign`

---

## ✔ Пример успешного ответа

```json
{
  "data": {
    "order": {
      "certificateRequests": {
        "csr": "-----BEGIN CERTIFICATE REQUEST-----\r\nMIICKzCCAZMCAQAwgYcxJjAkBgNVBAMMHdCk0JjQm9Ce0JfQntCkINCi0JDQotCs\r\n0K/QndCQMRcwFQYDVQQEDA7QpNCY0JvQntCX0J7QpDEYMBYGA1UEBRMPSUlOODMw\r\nNDA0MDAwMjM4MQswCQYDVQQGEwJLWjEdMBsGA1UEKgwU0JLQmNCi0JDQm9Cs0JXQ\r\nktCd0JAwgawwIwYJKoMOAwoBAQICMBYGCiqDDgMKAQECAgEGCCqDDgMKAQMDA4GE\r\nAASBgJpRo6Wy0h0NbZCyCj1BJed2cnAfx/tAMAODKdrfE8QCH5QXJfPTXfC2gzXf\r\n1FCZZJmd9n8fbrRZIw6P7V9oDA23WSYJRmSqOrBAaItYHfu7L6QPfMmyeC5FtGuF\r\nlT3gkCZvIuRY9GcbIlL+gUI9dgu3xRNN+5uKJqiH4NNuCcgKoFUwUwYJKoZIhvcN\r\nAQkOMUYwRDBCBgYqgw4DAwUEODA2Bggqgw4DAwUBAQwqQzpcVXNlcnNcbGFcRGVz\r\na3RvcFzQmNCY0KHQptCe0J1cdGVzdFx0YXQxMA4GCiqDDgMKAQECAwIFAAOBgQAu\r\nQWd2EmMscZqJy0ZkxuOn1TZly0s6+du67NZSGBlrj5fX1AhP+Hbf7SgZz4efn4ls\r\nsf891rlyR+KFpgiaSy5UPfg84mAhhOz5uwqXSup6HXMAqGOGTzTMItgqOQDmd2Vd\r\nshl5F97rY5bJYPXIpzYs4QypzkSPX6pk9PzRn6OhbA==\r\n-----END CERTIFICATE REQUEST-----",
        "requestTemplateName": "individ_sign_ng",
        "subjectDN": {
          "rdns": {
            "rdnName": "G",
            "value": "ВИТАЛЬЕВНА"
          }
        },
        "publicKey": "0481809a51a3a5b2d21d0d6d90b20a3d4125e77672701fc7fb4030038329dadf13c4021f941725f3d35df0b68335dfd4509964999df67f1f6eb459230e8fed5f680c0db75926094664aa3ab040688b581dfbbb2fa40f7cc9b2782e45b46b85953de090266f22e458f4671b2252fe81423d760bb7c5134dfb9b8a26a887e0d36e09c80a"
      },
      "referenceNumber": "17844565151117",
      "orderType": "ORDER_TYPE_ONLINE",
      "storageCode": "PKCS12",
      "digitalDoc": "false"
    },
    "status": {
      "isValid": "true"
    },
    "statement": {
      "mainTitleKz": "Қазақстан Республикасы Ұлттық куәландырушы орталығының тіркеу куәліктерін беруге өтініш (жеке тұлға үшін)",
      "mainTitleRu": "Заявление на выдачу регистрационных свидетельств Национального удостоверяющего центра Республики Казахстан (для физического лица)",
      "referenceNumber": {
        "labelKz": "Бірегей нөмір",
        "labelRu": "Уникальный номер заявки",
        "value": "17844565151117"
      },
      "referenceTemplate": {
        "labelKz": "Үлгі",
        "labelRu": "Шаблон",
        "nameKz": "Жеке тұлға МЕМСТ 2015",
        "nameRu": "Физическое лицо ГОСТ 2015"
      },
      "applicantInfoTitleKz": "Өтінішкердің сәйкестендіргіш деректері",
      "applicantInfoTitleRu": "Идентификационные данные заявителя",
      "applicantInfo": {
        "labelKz": "ТАӘ",
        "labelRu": "ФИО",
        "nameKz": "ФИЛОЗОФ ТАТЬЯНА ВИТАЛЬЕВНА",
        "nameRu": "ФИЛОЗОФ ТАТЬЯНА ВИТАЛЬЕВНА"
      },
      "phoneNumber": {
        "labelKz": "Телефон",
        "labelRu": "Телефон",
        "value": ":  ______________________________________________________"
      },
      "subjectInfoTitleKz": "Тіркеу куәліктерін алуға арналған сұратымдағы субъектінің деректері",
      "subjectInfoTitleRu": "Данные субъекта в запросе на получение регистрационного свидетельства",
      "subjectInfo": {
        "labelKz": "Ел",
        "labelRu": "Страна",
        "value": "KZ"
      },
      "additionalInfoTitleKz": "Қосымша ақпарат",
      "additionalInfoTitleRu": "Дополнительная информация",
      "additionalInfo": {
        "labelKz": "Тіркеу куәлігінің әрекет ету мерзімі",
        "labelRu": "Срок действия регистрационного свидетельства",
        "nameKz": "1 жыл",
        "nameRu": "1 год"
      },
      "firstAgreementKz": "Қазақстан Республикасы Ұлттық куәландырушы орталығы жазылушыларының тіркеу куәліктерін пайдалану қағидаларымен (Certificate Practice Statement) және Қазақстан Республикасы Ұлттық куәландырушы орталығы жазылушыларының тіркеу куәліктерін пайдалану саясатымен (Certificate Policy) таныстым, қарсылығым жоқ",
      "firstAgreementRu": "С правилами применения регистрационных свидетельств подписчиков Национального удостоверяющего центра Республики Казахстан (Certificate Practice Statement) и политикой применения регистрационных свидетельств подписчиков Национального удостоверяющего центра Республики Казахстан (Certificate Policy) ознакомлен, возражений не имею",
      "secondAgreementKz": "Қазақстан Республикасы Ұлттық куәландырушы орталығының пайдаланушы келісімімен танысып шықтым. Мемлекеттік көрсетілетін қызметті алу және электрондық поштаға хабарлама алу мақсатында менің дербес деректерімді өңдеуге келісімімді растаймын.",
      "secondAgreementRu": "С пользовательским соглашением Национального удостоверяющего центра Республики Казахстан ознакомлен и подтверждаю свое согласие на обработку моих персональных данных в целях получения государственной услуги и получения уведомлений на электронную почту.",
      "signatureInfoTitleKz": "Электрондық цифрлық қолтаңбаға тиісті жабық кілтін құру үшін пайдаланылатын электрондық цифрлық қолтаңбаның құралдары туралы деректер, электрондық цифрлық қолтаңба алгоритмі стандартының және ашық кілт ұзындығының анықтамасы",
      "signatureInfoTitleRu": "Данные о средствах электронной цифровой подписи, используемых для создания соответствующего закрытого ключа электронной цифровой подписи, обозначение стандарта алгоритма электронной цифровой подписи и длины открытого ключа",
      "certInfo": {
        "labelKz": "Аутентификацияның ашық кілті (ұзындығы - 512 бит, МЕМСТ 34.310-2004):",
        "labelRu": "Открытый ключ аутентификации (длина - 512 бит, ГОСТ 34.310-2004):",
        "value": "0481809a51a3a5b2d21d0d6d90b20a3d4125e77672701fc7fb4030038329dadf13c4021f941725f3d35df0b68335dfd4509964999df67f1f6eb459230e8fed5f680c0db75926094664aa3ab040688b581dfbbb2fa40f7cc9b2782e45b46b85953de090266f22e458f4671b2252fe81423d760bb7c5134dfb9b8a26a887e0d36e09c80a"
      },
      "bottomAdditionalInfo": {
        "labelKz": "Қосымша ақпаратқа арналған орын:",
        "labelRu": "Место для дополнительной информации:",
        "nameKz": "",
        "nameRu": ""
      },
      "referenceDate": {
        "labelKz": "«Азаматтарға арналған үкімет «Мемлекеттік корпорациясы» коммерциялық емес акционерлік қоғамына немесе көрсетілетін қызметті берушіге жүгінген күні:",
        "labelRu": "Дата обращения в некоммерческое акционерное общество «Государственная корпорация «Правительства для граждан» или к услугодателю:",
        "nameKz": "«____» ______________________ 20___ жыл",
        "nameRu": "«____» ______________________ 20___ год"
      },
      "signatureLabel": {
        "labelKz": "Жеке тұлғаның (жеке тұлға өкілінің) қолы",
        "labelRu": "Подпись физического лица (представителя физического лица)",
        "nameKz": "",
        "nameRu": ""
      }
    }
  },
  "applicationId": "052547529941",
  "confirmOrderSign": "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?><order xmlns=\"http://pki.gov.kz/api/ws/iiscon/wsdl\"><certificateRequests><csr>-----BEGIN CERTIFICATE REQUEST-----\nMIICKzCCAZMCAQAwgYcxJjAkBgNVBAMMHdCk0JjQm9Ce0JfQntCkINCi0JDQotCs\n0K/QndCQMRcwFQYDVQQEDA7QpNCY0JvQntCX0J7QpDEYMBYGA1UEBRMPSUlOODMw\nNDA0MDAwMjM4MQswCQYDVQQGEwJLWjEdMBsGA1UEKgwU0JLQmNCi0JDQm9Cs0JXQ\nktCd0JAwgawwIwYJKoMOAwoBAQICMBYGCiqDDgMKAQECAgEGCCqDDgMKAQMDA4GE\nAASBgJpRo6Wy0h0NbZCyCj1BJed2cnAfx/tAMAODKdrfE8QCH5QXJfPTXfC2gzXf\n1FCZZJmd9n8fbrRZIw6P7V9oDA23WSYJRmSqOrBAaItYHfu7L6QPfMmyeC5FtGuF\nlT3gkCZvIuRY9GcbIlL+gUI9dgu3xRNN+5uKJqiH4NNuCcgKoFUwUwYJKoZIhvcN\nAQkOMUYwRDBCBgYqgw4DAwUEODA2Bggqgw4DAwUBAQwqQzpcVXNlcnNcbGFcRGVz\na3RvcFzQmNCY0KHQptCe0J1cdGVzdFx0YXQxMA4GCiqDDgMKAQECAwIFAAOBgQAu\nQWd2EmMscZqJy0ZkxuOn1TZly0s6+du67NZSGBlrj5fX1AhP+Hbf7SgZz4efn4ls\nsf891rlyR+KFpgiaSy5UPfg84mAhhOz5uwqXSup6HXMAqGOGTzTMItgqOQDmd2Vd\nshl5F97rY5bJYPXIpzYs4QypzkSPX6pk9PzRn6OhbA==\n-----END CERTIFICATE REQUEST-----\n</csr><requestTemplateName>individ_sign_ng</requestTemplateName><subjectDN><rdns><rdnName>C</rdnName><value>KZ</value></rdns><rdns><rdnName>SERIALNUMBER</rdnName><value>IIN830404000238</value></rdns><rdns><rdnName>SURNAME</rdnName><value>ФИЛОЗОФ</value></rdns><rdns><rdnName>CN</rdnName><value>ФИЛОЗОФ ТАТЬЯНА</value></rdns><rdns><rdnName>G</rdnName><value>ВИТАЛЬЕВНА</value></rdns></subjectDN><publicKey>0481809a51a3a5b2d21d0d6d90b20a3d4125e77672701fc7fb4030038329dadf13c4021f941725f3d35df0b68335dfd4509964999df67f1f6eb459230e8fed5f680c0db75926094664aa3ab040688b581dfbbb2fa40f7cc9b2782e45b46b85953de090266f22e458f4671b2252fe81423d760bb7c5134dfb9b8a26a887e0d36e09c80a</publicKey></certificateRequests><referenceNumber>17844565151117</referenceNumber><orderType>ORDER_TYPE_ONLINE</orderType><storageCode>PKCS12</storageCode><digitalDoc>false</digitalDoc><applicationId>052547529941</applicationId><applicantNumber>7 (777) 777-77-77</applicantNumber><savedPhoto>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</savedPhoto><digitalDoc>false</digitalDoc><applicationType>scan</applicationType></order>",
  "denyOrderSign": "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?><order xmlns=\"http://pki.gov.kz/api/ws/iiscon/wsdl\"><certificateRequests><csr>-----BEGIN CERTIFICATE REQUEST-----\nMIICKzCCAZMCAQAwgYcxJjAkBgNVBAMMHdCk0JjQm9Ce0JfQntCkINCi0JDQotCs\n0K/QndCQMRcwFQYDVQQEDA7QpNCY0JvQntCX0J7QpDEYMBYGA1UEBRMPSUlOODMw\nNDA0MDAwMjM4MQswCQYDVQQGEwJLWjEdMBsGA1UEKgwU0JLQmNCi0JDQm9Cs0JXQ\nktCd0JAwgawwIwYJKoMOAwoBAQICMBYGCiqDDgMKAQECAgEGCCqDDgMKAQMDA4GE\nAASBgJpRo6Wy0h0NbZCyCj1BJed2cnAfx/tAMAODKdrfE8QCH5QXJfPTXfC2gzXf\n1FCZZJmd9n8fbrRZIw6P7V9oDA23WSYJRmSqOrBAaItYHfu7L6QPfMmyeC5FtGuF\nlT3gkCZvIuRY9GcbIlL+gUI9dgu3xRNN+5uKJqiH4NNuCcgKoFUwUwYJKoZIhvcN\nAQkOMUYwRDBCBgYqgw4DAwUEODA2Bggqgw4DAwUBAQwqQzpcVXNlcnNcbGFcRGVz\na3RvcFzQmNCY0KHQptCe0J1cdGVzdFx0YXQxMA4GCiqDDgMKAQECAwIFAAOBgQAu\nQWd2EmMscZqJy0ZkxuOn1TZly0s6+du67NZSGBlrj5fX1AhP+Hbf7SgZz4efn4ls\nsf891rlyR+KFpgiaSy5UPfg84mAhhOz5uwqXSup6HXMAqGOGTzTMItgqOQDmd2Vd\nshl5F97rY5bJYPXIpzYs4QypzkSPX6pk9PzRn6OhbA==\n-----END CERTIFICATE REQUEST-----\n</csr><requestTemplateName>individ_sign_ng</requestTemplateName><subjectDN><rdns><rdnName>C</rdnName><value>KZ</value></rdns><rdns><rdnName>SERIALNUMBER</rdnName><value>IIN830404000238</value></rdns><rdns><rdnName>SURNAME</rdnName><value>ФИЛОЗОФ</value></rdns><rdns><rdnName>CN</rdnName><value>ФИЛОЗОФ ТАТЬЯНА</value></rdns><rdns><rdnName>G</rdnName><value>ВИТАЛЬЕВНА</value></rdns></subjectDN><publicKey>0481809a51a3a5b2d21d0d6d90b20a3d4125e77672701fc7fb4030038329dadf13c4021f941725f3d35df0b68335dfd4509964999df67f1f6eb459230e8fed5f680c0db75926094664aa3ab040688b581dfbbb2fa40f7cc9b2782e45b46b85953de090266f22e458f4671b2252fe81423d760bb7c5134dfb9b8a26a887e0d36e09c80a</publicKey></certificateRequests><referenceNumber>17844565151117</referenceNumber><orderType>ORDER_TYPE_ONLINE</orderType><storageCode>PKCS12</storageCode><digitalDoc>false</digitalDoc><applicationId>052547529941</applicationId><applicantNumber>7 (777) 777-77-77</applicantNumber><digitalDoc>false</digitalDoc><applicationType>scan</applicationType></order>",
  "digiSign": "<?xml version=\"1.0\" encoding=\"UTF-8\"?><application><applicationId>052547529941</applicationId><serviceTypeCode>C09-03</serviceTypeCode><registerOrgCode>MID-GER-MUNCHEN</registerOrgCode><executorOrgCode>MTC-CTSAT</executorOrgCode><smsCode>14</smsCode></application>"
}
```

---

# 📄 API: `/service/nca/forward`

API предназначено для отправки данных заявки на подтверждение или отказ с последующей генерацией XML и подписанием структур.

---

# 🔗 **Endpoint**

```
POST /service/nca/forward
Content-Type: application/json
```

Метод контроллера:

```java
@PostMapping(value = "/forward", produces = MediaType.APPLICATION_JSON_VALUE)
public @ResponseBody Object getInfoOneC(@RequestBody ForwardApplication request) throws Exception {
```

---

# 📌 **Назначение API**

Сервис принимает заявление, данные заявителя, документы и XML-блоки, которые необходимо подписать, затем формирует и отправляет XML на внешнюю систему.

Поддерживаются два сценария:

* **C09-03-03-4** — подтверждение заявки
* **C09-03-03-5** — отказ от заявки

---

# 📥 **Примеры входящих JSON**

## ✅ Подтверждение заявки (`C09-03-03-4`)

<details>

```json
{
  "applicationId": "052703418875",
  "registerApplication": {
    "applicationRegistrationData": {
      "registerOrgCode": "MID-GER-MUNCHEN",
      "registerEmployee": {
        "lastName": "Тестов",
        "firstName": "Тест",
        "middleName": "Тест",
        "position": "Оператор ЦОН",
        "iin": "123456789021"
      }
    }
  },
  "updateApplication": {
    "applicationData": {
      "applicant": {
        "person": {
          "iin": "990312350989",
          "lastName": "ЖУМАНИЁЗОВ",
          "firstName": "СЕВИНДИК",
          "middleName": "КУВОНДИКУГЛИ",
          "contacts": {
            "contact": [
              {
                "contactType": "CELL_PHONE",
                "contactData": "7 (777) 777-77-77",
                "contactPrefix": "",
                "isForNotification": false
              }
            ]
          },
          "identityDocument": {
            "docType": "IDENTITY_CARD",
            "docNum": "051284930",
            "docDate": "2021-09-21+05:00",
            "docSource": "МВД РК",
            "expirationDate": "2031-09-20+05:00",
            "docName": "Удостоверение личности РК"
          },
          "dateOfBirth": "1999-03-12+05:00",
          "placeOfBirth": "КАЗАХСТАН КЫЗЫЛОРДИНСКАЯ ОБЛАСТЬ КЫЗЫЛОРДА",
          "gender": "M",
          "isForeigner": false,
          "citizenship": "398"
        }
      },
      "language": "RUS",
      "finalDocumentLang": "RUS",
      "documents": {
        "document": [
          {
            "documentId": "da9732c4-da69-4c34-a2ea-0d8cb6f5c658",
            "docTypeCode": "Z709",
            "copyType": "ELECTRONIC",
            "docName": "Заявление установленного образца"
          }
        ]
      },
      "data": {
        "digiSign": "...",
        "confirmOrder": {
          "digiSign": "..."
        }
      },
      "subServiceTypeCode": "C09-03-03-4"
    }
  }
}
```

</details>

---

## ❌ Отказ от заявки (`C09-03-03-5`)

<details>

```json
{
  "applicationId": "052631405066",
  "registerApplication": {
    "applicationRegistrationData": {
      "registerOrgCode": "MID-GER-MUNCHEN",
      "registerEmployee": {
        "lastName": "Тестов",
        "firstName": "Тест",
        "middleName": "Тест",
        "position": "Оператор ЦОН",
        "iin": "123456789021"
      }
    }
  },
  "updateApplication": {
    "applicationData": {
      "applicant": {
        "person": {
          "iin": "830404000238",
          "lastName": "ФИЛОЗОФ",
          "firstName": "ТАТЬЯНА",
          "middleName": "ВИТАЛЬЕВНА",
          "contacts": {
            "contact": [
              {
                "contactType": "CELL_PHONE",
                "contactData": "7 (777) 777-77-77",
                "contactPrefix": "",
                "isForNotification": false
              }
            ]
          },
          "identityDocument": {
            "docType": "FOREIGN_STATE_DOCUMENT",
            "docNum": "051284930",
            "docDate": "2021-09-21+05:00",
            "docSource": "Уполномоченный орган иностранного государства",
            "expirationDate": "2031-09-20+05:00",
            "docName": "Документ иностранного государства"
          },
          "dateOfBirth": "1983-04-04+05:00",
          "placeOfBirth": "КАЗАХСТАН КЫЗЫЛОРДИНСКАЯ ОБЛАСТЬ КЫЗЫЛОРДА",
          "gender": "Ж",
          "isForeigner": true,
          "citizenship": "580"
        }
      },
      "language": "RUS",
      "finalDocumentLang": "RUS",
      "documents": {
        "document": [
          {
            "documentId": "da9732c4-da69-4c34-a2ea-0d8cb6f5c658",
            "docTypeCode": "Z709",
            "copyType": "ELECTRONIC",
            "docName": "Заявление установленного образца"
          },
          {
            "documentId": "da9732c4-da69-4c34-a2ea-0d8cb6f5c658",
            "docTypeCode": "D344",
            "copyType": "ELECTRONIC",
            "docName": "Паспорт иностранного гражданина"
          }
        ]
      },
      "data": {
        "digiSign": "...",
        "denyOrder": {
          "digiSign": "..."
        }
      },
      "subServiceTypeCode": "C09-03-03-5"
    }
  }
}
```

</details>

---

# 🧾 Расшифровка полей запроса

## 1. Параметры из `application.properties`

| Поле          | Описание                  |
| ------------- | ------------------------- |
| **serviceId** | Идентификатор сервиса     |
| **senderId**  | Идентификатор отправителя |
| **password**  | Пароль                    |

---

## 2. Поля, приходящие в JSON

| Поле                   | Описание                                                                                                     |
|------------------------|--------------------------------------------------------------------------------------------------------------|
| **applicationId**      | Идентификатор заявки из API `service/nca/iiscon`                                                             |
| **registerOrgCode**    | Код регистрации заявки. Для прод: **MFA**, тест: `MID-GER-MUNCHEN`                                           |
| **docTypeCode**        | Тип документа:<br>• `D344` — паспорт иностранца<br>• `Z709` — заявление<br>• `D160` — удостоверение личности |
| **subServiceTypeCode** | <br>• `C09-03-03-4` — подтверждение<br>• `C09-03-03-5` — отказ                                               |
| **documentId**         | Это fileIdentifier, который возвращается при загрузке документа в сервис EDS_TEMP_FILES                      |

---

## 3. Поля, формируемые в XML

| Поле                         | Значение                                            |
| ---------------------------- | --------------------------------------------------- |
| **ServiceTypeCode**          | `C09-03`                                            |
| **smsCode**                  | `14`                                                |
| **registerMethod**           | `MOBILE`                                            |
| **orgCode**                  | `MFA`                                               |
| **systemCode**               | `052`                                               |
| **executorOrgCode**          | `MTC-CTSAT`                                         |
| **deliveryOrgCode**          | `MFA`                                               |
| **identityDocument.docDate** | `{identityDocumentDocDate}` — дата выдачи документа |
| **docDate**                  | текущее время формирования forward                  |

---

## 4. Подписи

| Поле                      | Описание                          |
| ------------------------- | --------------------------------- |
| **data.digiSign**         | XML, который требуется подписать  |
| **confirmOrder.digiSign** | XML для подписи при подтверждении |
| **denyOrder.digiSign**    | XML для подписи при отказе        |

⚠ **Важно:** это *не готовая подпись*, а XML-контент, который клиент должен подписать ЭЦП.

---

# 📤 Ответы API

## ▶ Успешное подтверждение

Содержит решение: *разрешено выпускать ЭЦП*.
Формат определяется внешней системой.

## ▶ Успешный отказ

Формат ожидается от внешней системы (на момент подготовки документа отсутствует).

---

# ⚙ Логика работы сервиса

1. Получает JSON.
2. Определяет тип операции по `subServiceTypeCode`:

    * `C09-03-03-4` → подтверждение
    * `C09-03-03-5` → отказ
3. Формирует XML-пакеты.
4. Передаёт клиенту XML на подпись (`digiSign`, `confirmOrder.digiSign` / `denyOrder.digiSign`).
5. После подписи — отправляет данные во внешнюю систему.
6. Возвращает ответ.

---



