package kz.arta.documentServices.listeners;

import kz.arta.documentServices.util.ConnectionPool;

import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import javax.ejb.ActivationConfigProperty;
import javax.ejb.MessageDriven;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.naming.NamingException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;

@MessageDriven(name = "DocflowDocument", activationConfig = {
        @ActivationConfigProperty(propertyName = "destinationType", propertyValue = "javax.jms.Queue"),
        @ActivationConfigProperty(propertyName = "destination", propertyValue = "java:jboss/queues/Synergy/DocflowDocument"),
        @ActivationConfigProperty(propertyName = "reconnectAttempts", propertyValue = "32"),
        @ActivationConfigProperty(propertyName = "reconnectInterval", propertyValue = "4000"),
        @ActivationConfigProperty(propertyName = "acknowledgeMode", propertyValue = "Auto-acknowledge") })
public class DocflowDocumentListener implements MessageListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(DocflowDocumentListener.class);

    public static final String EVENT_DOC_OPEN = "event.docflow.document.opened";
    public static final String EVENT_DOC_CLOSE = "event.docflow.document.closed";

    @javax.annotation.Resource
    private javax.ejb.MessageDrivenContext mdc;

    @Override
    public void onMessage(Message message) {
        try {
            String eventType = message.getStringProperty("api_event");
            String userID = message.getStringProperty("userId");
            String documentID = message.getStringProperty("documentId");

            if(documentID == null) {
                ObjectMapper mapper = new ObjectMapper();
                String json = message.getBody(String.class);
                Map<String, String> prop = mapper.readValue(json, Map.class);
                userID = prop.get("userId");
                documentID = prop.get("documentId");
                eventType = prop.get("api_event");
            }

            HashMap<String, String> info = getDocumentInfo(documentID);
            String dataUUID = info.get("dataUUID");

            if(null != eventType) {

                LOGGER.info("************************************");
                LOGGER.info("api_event: " + eventType);
                LOGGER.info("documentID: " + documentID);
                LOGGER.info("userID: " + userID);

                HashMap<String, String> openInfo = checkOpenDocument(documentID);
                LOGGER.info("[:: checkOpenDocument ::] " + openInfo);

                if(userID != null) {
                    if(EVENT_DOC_OPEN.equalsIgnoreCase(eventType)) {
                        if(openInfo.get("dataUUID") == null) {
                            addRow(documentID, dataUUID, userID);
                            LOGGER.info("добавляем запись в таблицу");
                        }
                    }
                    if(EVENT_DOC_CLOSE.equalsIgnoreCase(eventType)) {
                        if(openInfo.get("dataUUID") != null && userID.equalsIgnoreCase(openInfo.get("userID"))) {
                            deleteRow(documentID);
                            LOGGER.info("удаляем запись из таблицы");
                        }
                    }
                } else {
                    LOGGER.error("userID не должно быть NULL");
                }

                LOGGER.info("************************************");
            }
        } catch (Exception exc) {
            mdc.setRollbackOnly();
        }
    }



    private static void addRow(String documentID, String dataUUID, String userID) throws SQLException, NamingException {
        if (documentID == null) return;
        Connection con = null;
        try {
            con = ConnectionPool.getConnection();
            PreparedStatement statement = con.prepareStatement("INSERT INTO opened_documents (documentID, dataUUID, userID, date) VALUES (?, ?, ?, ?)");
            statement.setString(1, documentID);
            statement.setString(2, dataUUID);
            statement.setString(3, userID);
            statement.setLong(4, new Date().getTime());
            statement.executeUpdate();
        } catch (SQLException e) {
            LOGGER.error(e.getMessage(), e);
        } finally {
            ConnectionPool.close(con);
        }
    }

    public static int deleteRow(String documentID) throws SQLException, NamingException {
        if (documentID == null) return 0;
        Connection con = null;
        try {
            con = ConnectionPool.getConnection();
            PreparedStatement statement = con.prepareStatement("DELETE FROM opened_documents WHERE documentID = ?");
            statement.setString(1, documentID);
            return statement.executeUpdate();
        } catch (SQLException e) {
            LOGGER.error(e.getMessage(), e);
        } finally {
            ConnectionPool.close(con);
        }
        return 0;
    }

    public static int deleteAllRows() throws SQLException, NamingException {
        Connection con = null;
        try {
            con = ConnectionPool.getConnection();
            PreparedStatement statement = con.prepareStatement("DELETE FROM opened_documents");
            return statement.executeUpdate();
        } catch (SQLException e) {
            LOGGER.error(e.getMessage(), e);
        } finally {
            ConnectionPool.close(con);
        }
        return 0;
    }

    public static HashMap<String, String> checkOpenDocument(String documentID) throws SQLException, NamingException {
        if (documentID == null) return null;
        Connection con = null;
        try {
            con = ConnectionPool.getConnection();
            PreparedStatement ps = con.prepareStatement("SELECT dataUUID, userID, date FROM opened_documents WHERE documentID = ?");
            ps.setString(1, documentID);
            ResultSet res = ps.executeQuery();
            HashMap<String, String> result = new HashMap<>();

            if (res.next()) {
                result.put("documentID", documentID);
                result.put("dataUUID", res.getString("dataUUID"));
                result.put("userID", res.getString("userID"));
                result.put("date", res.getString("date"));
            }
            return result;
        } catch (SQLException e) {
            LOGGER.error(e.getMessage(), e);
        } finally {
            ConnectionPool.close(con);
        }
        return null;
    }

    public static ArrayList getAllDocuments() throws SQLException, NamingException {
        Connection con = null;
        try {
            con = ConnectionPool.getConnection();
            PreparedStatement ps = con.prepareStatement("SELECT documentID, dataUUID, userID, date FROM opened_documents");
            ResultSet res = ps.executeQuery();
            ArrayList<HashMap<String, String>> docs = new ArrayList<>();
            while (res.next()) {
                HashMap<String, String> row = new HashMap<>();
                row.put("documentID", res.getString("documentID"));
                row.put("dataUUID", res.getString("dataUUID"));
                row.put("userID", res.getString("userID"));
                row.put("date", res.getString("date"));
                docs.add(row);
            }
            return docs;
        } catch (SQLException e) {
            LOGGER.error(e.getMessage(), e);
        } finally {
            ConnectionPool.close(con);
        }
        return null;
    }

    public static HashMap<String, String> getDocumentInfo(String documentID) throws SQLException, NamingException {
        if (documentID == null) return null;

        Connection con = null;
        try {
            con = ConnectionPool.getConnection();
            PreparedStatement ps = con.prepareStatement("SELECT\n" +
                    "rg.documentID, rg.asfDataID,\n" +
                    "r.registryID, r.code registryCode,\n" +
                    "r.formid formID, ad.code formCode\n" +
                    "FROM registry_documents rg\n" +
                    "LEFT JOIN registries r ON r.registryID = rg.registryID\n" +
                    "LEFT JOIN asf_definition ad ON ad.uuid = r.formid\n" +
                    "WHERE rg.documentID = ?");

            ps.setString(1, documentID);
            ResultSet res = ps.executeQuery();

            String dataUUID = null;
            String registryID = null;
            String registryCode = null;
            String formID = null;
            String formCode = null;

            if (res.next()) {
                dataUUID = res.getString("asfDataID");
                registryID = res.getString("registryID");
                registryCode = res.getString("registryCode");
                formID = res.getString("formID");
                formCode = res.getString("formCode");
            }

            HashMap<String, String> result = new HashMap<>();
            result.put("documentID", documentID);
            result.put("dataUUID", dataUUID);
            result.put("registryID", registryID);
            result.put("registryCode", registryCode);
            result.put("formID", formID);
            result.put("formCode", formCode);

            return result;
        } catch (SQLException e) {
            LOGGER.error(e.getMessage(), e);
        } finally {
            ConnectionPool.close(con);
        }
        return null;
    }
}