package nce.integration.synergy.service;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import feign.Response;
import feign.gson.GsonDecoder;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import nce.integration.synergy.Const;
import nce.integration.synergy.entity.*;
import nce.integration.synergy.repo.FeignClientRepo;
import nce.integration.synergy.repo.SynergyClientRepo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Type;

import org.json.JSONObject;

import java.net.URI;
import java.net.URISyntaxException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

@Service
@RequiredArgsConstructor
@EnableScheduling
public class NceIntegrationService {
    private final FeignClientRepo feignClientRepo;
    private final SynergyClientRepo synergyClientRepo;
    private static final Gson gson = new Gson();

    private static SimpleDateFormat keyFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
    private static SimpleDateFormat valueFormat = new SimpleDateFormat("dd.MM.yyyy HH:mm");

    private static final Logger LOGGER = LoggerFactory.getLogger(NceIntegrationService.class);

    @Scheduled(cron = "${cronUpdate}")
    public void getCovidData() throws IOException, URISyntaxException, ParseException {
        DictionaryData dictionaryData = getPCRLaboratory();
        List<String> binList = dictionaryData.getALLBin();      // Лист Бинов из справочника Лаборатории по ПЦР
        List<String> binListNames = dictionaryData.getALLBinName();
        String token = getTokenData();                    // Токен от NCE

        List<UnitsData> unitsData = getUnitsData(token);  // не распарсенные данные

        HashMap<String, FormatUnitData> formatUnitDataHashMap = new HashMap<>();
        if (unitsData.size() > 0) {
            formatUnitDataHashMap = parseUnitsData(unitsData, binList, binListNames);
        }

        for (String key : formatUnitDataHashMap.keySet()) {
            String dataUUID = createRecord();
            LOGGER.info("NCE Integration: Record created with dataUUID = " + dataUUID);
            if (dataUUID != null) {
                MergeDataToRecord(dataUUID, formatUnitDataHashMap.get(key));
            }

            dataUUID = ActivateRecord(dataUUID);

            if (dataUUID != null) {
                LOGGER.info("NCE Integration: Record activated with dataUUID = " + dataUUID);
            } else {
                LOGGER.info("NCE Integration: Record not activated with dataUUID = " + dataUUID);
            }
        }
    }

    // functions with Record
    public String createRecord() throws URISyntaxException, IOException {
        String url = "?registryCode=ptsr-issledovaniya";
        Response response = this.feignClientRepo.getDictionaryData(Const.SYNERGY_AUTH, new URI(Const.SYNERGY_HOST + "/rest/api/registry/create_doc" + url));
        GsonDecoder gsonDecoder = new GsonDecoder();
        NewRecordData newRecordData = (NewRecordData) gsonDecoder.decode(response, NewRecordData.class);

        if (response != null) {
            response.close();
        }
        if (newRecordData.getErrorCode().equals("0")) {
            return newRecordData.getDataUUID();
        } else {
            return null;
        }
    }

    public void MergeDataToRecord(String dataUUID, FormatUnitData formatUnitData) throws IOException {
        MergeDataFormat mergeDataFormat = new MergeDataFormat(dataUUID);

        Integer formulaDone = formatUnitData.getFirst() + formatUnitData.getSecond();
        Integer numericInputTotal = formulaDone + formatUnitData.getAll_tests();
        mergeDataFormat.getData().add(new MergeDataFormat.MergeCmpData("numericinput_first", "numericinput", formatUnitData.getFirst().toString(), formatUnitData.getFirst().toString())); // numericinput_first
        mergeDataFormat.getData().add(new MergeDataFormat.MergeCmpData("numericinput_second", "numericinput", formatUnitData.getSecond().toString(), formatUnitData.getSecond().toString())); // numericinput_second
        mergeDataFormat.getData().add(new MergeDataFormat.MergeCmpData("numericinput_positive", "numericinput", formatUnitData.getPositive().toString(), formatUnitData.getPositive().toString())); // numericinput_positive
        mergeDataFormat.getData().add(new MergeDataFormat.MergeCmpData("numericinput_negative", "numericinput", formatUnitData.getNegative().toString(), formatUnitData.getNegative().toString())); // numericinput_negative
        mergeDataFormat.getData().add(new MergeDataFormat.MergeCmpData("formula_test_done", "numericinput", formulaDone.toString(), formulaDone.toString())); // formula_test_done
        mergeDataFormat.getData().add(new MergeDataFormat.MergeCmpData("date_create_data", "date", formatUnitData.getValueDate(), formatUnitData.getKeyDate())); // date_create_date
        mergeDataFormat.getData().add(new MergeDataFormat.MergeCmpData("listbox_organizations", "listbox", formatUnitData.getBinName(), formatUnitData.getBin())); // listbox_organizations
        mergeDataFormat.getData().add(new MergeDataFormat.MergeCmpData("numericinput_all_period_test_count", "numericinput", numericInputTotal.toString(), numericInputTotal.toString())); // numericinput_all_period_test_count

      Response response = synergyClientRepo.mergeCovidData(Const.SYNERGY_AUTH, mergeDataFormat);

        GsonDecoder gsonDecoder = new GsonDecoder();
        MergeResponse mergeResponse = (MergeResponse) gsonDecoder.decode(response, MergeResponse.class);

        if (response != null) {
            response.close();
        }
        if (!mergeResponse.getErrorCode().equals("0")) {
            LOGGER.error("Error code " + mergeResponse.getErrorCode() + ", merging data into form with UUID = " + mergeResponse.getUuid());
        } else {
            LOGGER.info("NCE Integration: Data merge to dataUUID = " + dataUUID + " successful");
        }
    }

    private String ActivateRecord(String dataUUID) throws IOException, URISyntaxException {
        Response response = this.feignClientRepo.getDictionaryData(Const.SYNERGY_AUTH, new URI(Const.SYNERGY_HOST + "/rest/api/registry/activate_doc?dataUUID=" + dataUUID));
        GsonDecoder gsonDecoder = new GsonDecoder();
        NewRecordData newRecordData = (NewRecordData) gsonDecoder.decode(response, NewRecordData.class);

        if (response != null) {
            response.close();
        }

        if (newRecordData.getErrorCode().equals("0")) {
            return newRecordData.getDataUUID();
        } else {
            return null;
        }
    }


    private List<UnitsData> getUnitsData(String token) throws IOException {
        Response response = feignClientRepo.getCovidData("Bearer " + token);
        //GsonDecoder gsonDecoder = new GsonDecoder();
        //CovidData covidData = (CovidData) gsonDecoder.decode(response, CovidData.class);

        GsonBuilder gsonBuilder = new GsonBuilder();
        CovidData covidData = gsonBuilder.setDateFormat("yyyy-MM-dd'T'HH:mm:ss").create().fromJson(response.body().asReader(), CovidData.class);

        if (response != null) {
            response.close();
        }

        return covidData.getResults();
    }

    private HashMap<String, FormatUnitData> parseUnitsData(List<UnitsData> unitsDataList, List<String> binList, List<String> binListNames) throws ParseException, IOException, URISyntaxException {
        Date yesterday = new Date(System.currentTimeMillis() - 24 * 60 * 60 * 1000);
        HashMap<String, FormatUnitData> formatUnitDataHashMap = new HashMap<>();
        for (UnitsData unitsData : unitsDataList) {
            for (String key : unitsData.getUnits().keySet()) {
                //Date date = formatter.parse(unitsData.getUnits().get(key).getDay());
                String bin = unitsData.getUnits().get(key).getBin();
                Date date = unitsData.getUnits().get(key).getDay();
                String regionISO = unitsData.getUnits().get(key).getRegionISO();
                String probe = unitsData.getUnits().get(key).getProbe();
                Integer count = unitsData.getUnits().get(key).getCount();
                Boolean result = unitsData.getUnits().get(key).getResult();

                if (binList.contains(bin) && yesterday.getYear() == date.getYear() && yesterday.getMonth() == date.getMonth() && yesterday.getDate() == date.getDate() && regionISO.equals(Const.REGION_ISO)) {

                    if (formatUnitDataHashMap.containsKey(bin)) {
                        FormatUnitData formatUnitData = formatUnitDataHashMap.get(bin);
                        formatUnitData.setKeyDate(keyFormat.format(date));
                        formatUnitData.setValueDate(valueFormat.format(date));

                        if (probe.equals("Primary")) {
                            formatUnitData.setFirst(formatUnitData.getFirst() + count);
                        } else if (probe.equals("Repeatedly")) {
                            formatUnitData.setSecond(formatUnitData.getSecond() + count);
                        } else if (probe.equals("Treatment Control")) {
                            formatUnitData.setSecond(formatUnitData.getSecond() + count);
                        }

                        if (result) {
                            formatUnitData.setPositive(formatUnitData.getPositive() + count);
                        } else {
                            formatUnitData.setNegative(formatUnitData.getNegative() + count);
                        }

                    } else {
                        FormatUnitData formatUnitData = new FormatUnitData();
                        formatUnitData.setBin(bin);
                        formatUnitData.setBinName(binListNames.get(binList.indexOf(bin)));
                        formatUnitData.setKeyDate(keyFormat.format(date));
                        formatUnitData.setValueDate(valueFormat.format(date));

                        if (probe.equals("Primary")) {
                            formatUnitData.setFirst(count);
                        } else if (probe.equals("Repeatedly")) {
                            formatUnitData.setSecond(count);
                        } else if (probe.equals("Treatment Control")) {
                            formatUnitData.setSecond(count);
                        }

                        if (result) {
                            formatUnitData.setPositive(count);
                        } else {
                            formatUnitData.setNegative(count);
                        }

                        DailyCapacity dailyCapacity = PowerOnDay(bin);
                        if (dailyCapacity.getResult().size() > 0) {
                            try {
                                formatUnitData.setPower_on_day(dailyCapacity.getResult().get(0).getFieldKey().getNumericinput_power_on_day());
                            } catch (Exception err) {
                                LOGGER.error("Error in parsing class err = " + err.toString());
                            }
                        }

                        AllTestDone allTestDone = AllTestDone(bin);

                        if (allTestDone.getResult().size() > 0) {
                            try {
                                formatUnitData.setAll_tests(allTestDone.getResult().get(0).getFieldKey().getNumericinput_all_period_test_count());
                            } catch (Exception err) {
                                LOGGER.error("Error in parsing class err = " + err.toString());
                            }
                        }
                        formatUnitDataHashMap.put(bin, formatUnitData);
                    }
                }

            }
        }
        return formatUnitDataHashMap;
    }

    private String getTokenData() {
        AuthData authData = new AuthData(Const.NCE_USERNAME, Const.NCE_PASSWORD);

        Type type = new TypeToken<TokenData>() {
        }.getType();
        Response response = feignClientRepo.getNceToken(authData);
        TokenData tokenData = null;
        try {
            try (BufferedReader buffer = new BufferedReader(new InputStreamReader(response.body().asInputStream()))) {
                String resp = buffer.lines().collect(Collectors.joining("\n"));
                tokenData = gson.fromJson(resp, type);
            }
        } catch (IOException ex) {
            throw new RuntimeException("Failed to process response body.", ex);
        }

        if (tokenData.getErrorCode() == null)
            return tokenData.getToken();
        else
            return null;
    }


    // Additional data
    private DictionaryData getPCRLaboratory() throws URISyntaxException, IOException {
        Response response = this.feignClientRepo.getDictionaryData(Const.SYNERGY_AUTH, new URI(Const.SYNERGY_HOST + "/rest/api/dictionaries/dict_pcr_laboratories?getColumns=false"));
        GsonDecoder gsonDecoder = new GsonDecoder();
        DictionaryData dictionaryData = (DictionaryData) gsonDecoder.decode(response, DictionaryData.class);

        if (response != null) {
            response.close();
        }

        return dictionaryData;
    }

    private DailyCapacity PowerOnDay(String bin) throws IOException, URISyntaxException {
        String url = "?registryCode=registry_information_on_the_daily_capacity_of_the_laboratory&fields=numericinput_power_on_day&fields=listbox_organizations";
        url = url + "&field=listbox_organizations&condition=CONTAINS&key=" + bin;
        Response response = this.feignClientRepo.getDictionaryData(Const.SYNERGY_AUTH, new URI(Const.SYNERGY_HOST + "/rest/api/registry/data_ext" + url));
        GsonDecoder gsonDecoder = new GsonDecoder();
        DailyCapacity dailyCapacity = (DailyCapacity) gsonDecoder.decode(response, DailyCapacity.class);

        if (response != null) {
            response.close();
        }
        return dailyCapacity;
    }

    private AllTestDone AllTestDone(String bin) throws IOException, URISyntaxException {
        String url = "?registryCode=ptsr-issledovaniya&fields=numericinput_all_period_test_count&fields=listbox_organizations";
        url = url + "&pageNumber=0&countInPart=1&field=listbox_organizations&condition=CONTAINS&key=" + bin;
        Response response = this.feignClientRepo.getDictionaryData(Const.SYNERGY_AUTH, new URI(Const.SYNERGY_HOST + "/rest/api/registry/data_ext" + url));
        GsonDecoder gsonDecoder = new GsonDecoder();
        AllTestDone allTestDone = (AllTestDone) gsonDecoder.decode(response, AllTestDone.class);

        if (response != null) {
            response.close();
        }
        return allTestDone;
    }

    @Data
    public class MergeResponse {
        private String errorCode;
        private String uuid;
    }

}
