//
//  UITextBoxTableViewCell.swift
//  kt
//
//  Created by Admin on 10.12.16.
//  Copyright © 2016 Alexandr Zalutskiy. All rights reserved.
//

import Foundation


class TextBoxCellData : TitleValueCellData {
    
    internal var placeholder : String = ""
    internal var keyboardType : UIKeyboardType = UIKeyboardType.Default
    
    internal init(title:String, value:String, placeholder:String, onValidate:(AnyObject?) -> Bool) {
        super.init(title: title, value: value)
        self.placeholder = placeholder
        self.onValidate = onValidate
    }
    
    override internal var cellType: CellType { return CellType.TextBoxCell}
    
    override func checkValues() -> Bool {
        return onValidate(value)
    }
}

@objc protocol UITextFieldTableViewCellDelegate {
    
    optional func textFieldDidBeginEditing(textField : UITextField, cell : UIBaseTableViewCell);
    optional func textFieldDidEndEditing(textField : UITextField, cell : UIBaseTableViewCell);
    
}

class UITextBoxTableViewCell : UIBaseTableViewCell, UITextFieldDelegate {
    
    internal var delegate : UITextFieldTableViewCellDelegate?
    
    @IBOutlet weak var textField: UITextField!
    @IBOutlet weak var title: UILabel!
    
    private var myCellData : TextBoxCellData!
    var lblPrefix: UILabel?
    
    override func awakeFromNib() {
        super.awakeFromNib()
        textField.applyUITextFieldStyle()       
    }
    
    override func fillWithCellData(cellData: CellData) {
        super.fillWithCellData(cellData)
        myCellData = cellData as! TextBoxCellData
        textField.text = myCellData.value
        textField.placeholder = myCellData.placeholder
        textField.delegate = self
        textField.keyboardType = myCellData.keyboardType
        
        if (textField.keyboardType == UIKeyboardType.PhonePad) {
            lblPrefix = KeyboardUtil.changeKeyboardType(UIKeyboardType.PhonePad, textField: textField)
        }
        
        title.text = myCellData.title
        
        cellData.addObserver("titleText") { (currentValue) in
            self.title.text = currentValue
        }
        cellData.addObserver("valueText") { (currentValue) in
            self.textField.text = currentValue
        }
    }  
    
    internal func RenderPrefix() {
        if (lblPrefix != nil && textField.text?.characters.count == 0) {
            lblPrefix?.textColor = Color.PlaceholderGrayColor
        } else {
            ShowPrefixAsBlackColor()
        }
    }
    
    internal func ShowPrefixAsBlackColor() {
        lblPrefix?.textColor = Color.TitleBlackColor
    }
    
    func textFieldDidBeginEditing(textField: UITextField) {
        delegate?.textFieldDidBeginEditing!(textField, cell: self)
        textField.setBottomBorder(Color.BlueButtonColor)
        RenderPrefix()
    }
    
    func textFieldDidEndEditing(textField: UITextField) {
        delegate?.textFieldDidEndEditing!(textField, cell: self)
        textField.setBottomBorder(Color.GrayTableCellSeparator)
        RenderPrefix()
    }
    
    func textFieldShouldReturn(textField: UITextField) -> Bool {
        self.endEditing(true)
        return false
    }
    
    override func updateCellData() {
        myCellData.value = textField.text!
    }
    
    func textField(textField: UITextField, shouldChangeCharactersInRange range: NSRange, replacementString string: String) -> Bool {
        guard let text = textField.text else { return true }
        let newLength = text.utf16.count + string.utf16.count - range.length
        return newLength <= 10
    }
    
}
